{$N+}
program RADIO_HORIZON;  {j.pike 14/294/2100}

{This routine uses the bi-exponential model of radio refractivity to obtain a
value of radio wave curvature at any altitude, and with it, determine whether
an intervening landmark projects into the propagation path between two given
points.  Since several computations involve the difference between very large
similar numbers, the accuracy (19-20 digits) of extended numbers is needed.
Variable names with underscore follow the model derivation}

uses crt;

var
  radian: extended;

function EARCCOS (x: extended): extended;  {in radians}
  var a:  extended;
  begin
    a:= (-x * x + 1);
    if a = 0.0 then
      a:= 1.0e-30;  {prevents divide by 0 error below}
    earccos:= pi / 2.0 - arctan (x / sqrt (a));
  end;

function ESNE (x: extended): extended;  {sine in degrees}
  begin
    x:= x / radian;
    esne:= sin (x);
  end;

function ECSN (x: extended): extended;  {cosine in degrees}
  begin
    x:= x / radian;
    ecsn:= cos (x);
  end;

function EACS (x: extended): extended;  {arccosine in degrees}
  begin
    eacs:= radian * earccos (x)
  end;

procedure INPUT (s: string; var x: real);
  var z: string [20]; c: integer;
  begin
    write (s, ' ( ', x:12:4, ' ) ');
    readln (z);
    if z = '' then
      exit;
    val (z, x, c);
    if c = 0 then
      exit
    else
      writeln ('Error at position ', c);
  end;

procedure INIT_HORIZON(var  se, od, oe, rd, re: real);
  begin
    se:= 7836.0;  {Mt. Hood Timberline repeater from Table Rock L.O.}
    od:= 98.1;  {Rancheria Rock obstruction}
    oe:= 4400.0;
    rd:= 180.2;
    re:= 7000.0;
  end;

procedure INPUT_HORIZON(var  se, od, oe, rd, re: real);
  begin
    input ('    Station elevation, feet     ', se);
    input ('    Obstruction distance, miles ', od);
    input ('    Obstruction elevation, ft   ', oe);
    input ('    Repeater distance, mi       ', rd);
    input ('    Repeater elevation, ft      ', re);
  end;

procedure COMPUTE_HORIZON(stnelev, obstrdist, obstrelev, rptrdist,
    rptrelev: real; var clearance, factor, ngradient, rgradient: real);
  const
    eradius = 20912073.0;  {earth radius in feet (6374 km)}
    a0 = 2.735;  {ray radius coefficients derived from Bean & Dutton's}
    a1 = 2.945e-4;  {bi-exponential model: "temperate" & "contiguous states"}
  var
    radius, wradius: real;
    a_, b_, c_, d_, e_, f_, g_, j_, p_, u_, v_, w_: extended;
  begin
    factor:= a0+a1*(stnelev+rptrelev)/2.0;
    wradius:= factor*eradius;
    ngradient:= 1.0e6/(wradius/3280.84);
    a_:= obstrdist*5280.0;
    b_:= (rptrdist-obstrdist)*5280.0;
    u_:= (a_/eradius)*radian;
    v_:= (b_/eradius)*radian;
    p_:= (rptrdist/2.0)*5280.0;
    c_:= wradius-sqrt(sqr(wradius)-sqr(p_));
    j_:= (p_-abs(p_-a_))/p_;
    d_:= c_*sqr(j_);
    w_:= sqrt(sqr(eradius+stnelev)+sqr(eradius+rptrelev)-
      2.0*(eradius+stnelev)*(eradius+rptrelev)*ecsn(u_+v_));
    e_:= eacs((sqr(w_)+sqr(eradius+rptrelev)-
      sqr(eradius+stnelev))/(2.0*w_*(eradius+rptrelev)));
    f_:= 180.0-e_-u_-v_;
    g_:= w_/(esne(u_)/esne(f_)+esne(v_)/esne(e_))-eradius;
    clearance:= g_+d_-obstrelev;
    c_:= (obstrelev-g_)/sqr(j_);
    radius:= (sqr(c_)+sqr(rptrdist*5280.0/2.0))/(2.0*c_);
    rgradient:= 1.0e6/(radius/3280.84);
  end;

procedure DISPLAY_HORIZON(clearance, factor, ngradient, rgradient: real);
  var
    path: string[50];
  begin
    if clearance < 0.0 then
      path:= 'The obstruction projects into the path by '
    else
      path:= 'The path lies above the obstruction by ';
    writeln ('      ', path, abs(clearance):1:1, ' ft');
    writeln ('      The radio wave radius of curvature is ',
      factor:1:1, ' earth radii');
    writeln ('      The refractivity gradient is -',
      ngradient:1:1, ' N-units/km');
    if clearance < 0.0 then
      writeln ('      The gradient required to clear the obstruction is -',
        rgradient:1:1, ' N-units/km');
  end;

var
  clr, fac, ngrad, obstrd, obstre, rgrad, rptrd, rptre, stne: real;
  q: char;

begin
  clrscr;
  gotoxy (3,2);
  writeln ('RADIO HORIZON CALCULATIONS');
  radian:= 180.0/pi;
  INIT_HORIZON(stne, obstrd, obstre, rptrd, rptre);
  repeat
    INPUT_HORIZON(stne, obstrd, obstre, rptrd, rptre);
    COMPUTE_HORIZON(stne, obstrd, obstre, rptrd, rptre, clr, fac,
      ngrad, rgrad);
    DISPLAY_HORIZON(clr, fac, ngrad, rgrad);
    write ('                    <Enter> or <Q>uit ');
    q:= upcase (readkey);
    if q = 'Q' then
      exit;
    writeln;  writeln;
  until false;
end.
